
-- SQL setup for Referral Earning Website

CREATE DATABASE IF NOT EXISTS referral_site CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE referral_site;

-- users table
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  email VARCHAR(150) NOT NULL UNIQUE,
  password VARCHAR(255) NOT NULL,
  referral_code VARCHAR(50) NOT NULL UNIQUE,
  used_referral_code VARCHAR(50) DEFAULT NULL,
  balance DECIMAL(12,2) DEFAULT 0,
  is_active TINYINT(1) DEFAULT 0,
  profile_pic VARCHAR(255) DEFAULT NULL,
  uid VARCHAR(50) DEFAULT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- referrals table (who referred who)
CREATE TABLE IF NOT EXISTS referrals (
  id INT AUTO_INCREMENT PRIMARY KEY,
  referrer_id INT NOT NULL,
  referee_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (referrer_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (referee_id) REFERENCES users(id) ON DELETE CASCADE
);

-- slot settings (admin-configured prices and capacities)
CREATE TABLE IF NOT EXISTS slot_settings (
  slot_number INT PRIMARY KEY,
  price DECIMAL(12,2) NOT NULL DEFAULT 10,
  capacity INT NOT NULL DEFAULT 3
);

-- user slots (which slots each user updated/activated)
CREATE TABLE IF NOT EXISTS user_slots (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  slot_number INT NOT NULL,
  activated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- deposits
CREATE TABLE IF NOT EXISTS deposits (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  method VARCHAR(50),
  amount DECIMAL(12,2),
  transaction_id VARCHAR(255),
  status ENUM('pending','approved','rejected') DEFAULT 'pending',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- withdraws
CREATE TABLE IF NOT EXISTS withdraws (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL,
  method VARCHAR(50),
  amount DECIMAL(12,2),
  account VARCHAR(255),
  status ENUM('pending','approved','rejected') DEFAULT 'pending',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- salary fund (admin fund)
CREATE TABLE IF NOT EXISTS salary_fund (
  id INT AUTO_INCREMENT PRIMARY KEY,
  amount DECIMAL(12,2) NOT NULL,
  source VARCHAR(255) DEFAULT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- settings
CREATE TABLE IF NOT EXISTS settings (
  id INT AUTO_INCREMENT PRIMARY KEY,
  site_name VARCHAR(255) DEFAULT 'Referral Site',
  bkash_number VARCHAR(100) DEFAULT '',
  nagad_number VARCHAR(100) DEFAULT '',
  min_deposit DECIMAL(12,2) DEFAULT 10,
  min_withdraw DECIMAL(12,2) DEFAULT 10
);

-- seed default settings row
INSERT INTO settings (site_name, bkash_number, nagad_number, min_deposit, min_withdraw)
SELECT 'FF BINFUN TOURNAMENT', '', '', 10, 10
WHERE NOT EXISTS (SELECT 1 FROM settings);

-- seed slot settings
INSERT INTO slot_settings (slot_number, price, capacity)
VALUES (1, 10.00, 3),
       (2, 50.00, 10),
       (3, 200.00, 50),
       (4, 1000.00, 999999)
ON DUPLICATE KEY UPDATE price=VALUES(price), capacity=VALUES(capacity);

-- create a starter system user (so new users can use referral code)
INSERT INTO users (name, email, password, referral_code, is_active, balance, uid)
SELECT 'System', 'system@local', '$2y$10$abcdefghijklmnopqrstuv', 'WELCOME123', 1, 0.00, 'SYS001'
WHERE NOT EXISTS (SELECT 1 FROM users WHERE referral_code='WELCOME123');
